/** How to label the triangles. 
 *  (1) Start with top row of hexagon, and work down, row by row. Total 2*_numTris rows.
 *  (2) Number of tris in each row is 2*_numTris+1,2*_numTris+1+3,..,4*_numTris-1,4*_numTris-1,..,2*_numTris+1.
 *  (3) Along each row in top half, odd ones have vertex pointing up, and even ones have vertex pointing down. 
 *  (4) Reverse (3) for bottom half. 
 *  (5) So each triangle has a row and a column index, determined by a double array index for _triangless.
 */ 

package sprites {

	import Data.Ray;
	import Data.TriangleData;
	
	import events.BeginRayEvent;
	import events.DestroyRayEvent;
	//import events.TriCountEvent;
	
	import flash.display.Graphics;
	import flash.display.Shape;
	import flash.display.Sprite;
	import flash.events.MouseEvent;
	
	import mx.controls.Alert;
	
	import util.Random;
	
	public class HexSprite extends Sprite {
		
		// Constants.
		private const RAY_LINE_THICK:Number = 4;
		private const RAY_LINE_COL:uint = 0xcc0000;
		private const RAY_ALPHA:Number = 0.5;
		private const ROOT3:Number = Math.sqrt(3);
//		private const SPOT_RAD_PROP:Number = ROOT3/6-0.01; // Spot radius = 0.2*_triSide.
		private const PI:Number = Math.PI;
		private const PROB_FILL:Number = 0.2; //Probability a triangle is filled.
		
		// Private variables.
		private var _size:Number = 50; // Width of hexagon.
		private var _numTris:uint = 3; // Number of triangles, counted from centre outwards.
		public function get numTris():uint { return _numTris; }
		private var _triSide:Number = 10; // Equal to (_size)/(2*_numTris).
		private var _triHeight:Number = 8; // Equal to _triSide*ROOT3_OVER2.
		private var _triangles:Array;
		public function get triangles():Array { return _triangles; }
		private var _spots:Array;
		private var _triangleSprite:Sprite = new Sprite(); //Holds triangles.
		public function get triangleSprite():Sprite { return _triangleSprite; }
		private var _spotsSprite:Sprite = new Sprite(); //Holds spots.
		public function get spotsSprite():Sprite { return _spotsSprite; }
		private var _rayShape:Shape = new Shape(); //Holds rays.
		public function get rayShape():Shape { return _rayShape; }
		private var _rays:Array = []; //Each element is a Ray.	
		private var _numberLabel:Boolean = false; //Either numbers or letters.
//		private var _disabled:Boolean = false; //True iff cannot use triangles.
//		private var _triCount:int = 0;
		
		
		/**
		 * Constructor
		 */ 
		
		public function HexSprite(numTris:int,numberLabel:Boolean) {
			
		//	_disabled = disabled;
			_numTris = numTris;
			_numberLabel = numberLabel;
			
			_triangles = new Array(_numTris*2);
			_spots = new Array(_numTris*6);
			
		//	trace("_disabled = "+_disabled);
		//	trace("_numTris = "+_numTris);
			
			addChild(_triangleSprite);
			addChild(_spotsSprite);
			createTriangles();
			createSpots();
			addChild(_rayShape);
			_rayShape.alpha = 0.5;
			
			//Listen for when a button pressed.
			addEventListener(BeginRayEvent.BEGIN_RAY,beginRayHandler);
			addEventListener(DestroyRayEvent.DESTROY_RAY,destroyRayHandler);
			
			
		}
		
		/**
		 * Initial creation functions. Once only.
		 */ 
		
		//Initialise the triangles array.
		private function createTriangles():void {
			var i:int, j:int,L:int, b:Boolean;
			for (i=0; i!=_numTris*2; i++) {
				L = 4*_numTris-Math.abs(2*_numTris-1-2*i);
				_triangles[i] = new Array(L);
				for (j=0; j!=L; j++) {
					b = ( (i<_numTris && j%2==0) || (i>=_numTris && j%2==1) );
					_triangles[i][j] = new Triangle(b);	
					_triangleSprite.addChild(_triangles[i][j]);
					_triangles[i][j].addEventListener(MouseEvent.CLICK,triangleClickHandler);
				}
			}
		}
		
		//Initialise spots.
		private function createSpots():void {
			var i:int,j:int,k:int,M:int,L:int = _numTris*6;
			
			for (i=0; i!=L; i++) {
				j = i%_numTris;
				if (i<_numTris) {
					_spots[i] = new Spot(i,0,2*j+1,"SE","SW",_numberLabel);
				} else if (i<2*_numTris) {
					M = 4*_numTris-Math.abs(2*_numTris-1-2*j)-1;
					_spots[i] = new Spot(i,j,M,"SW","W",_numberLabel);
				} else if (i<3*_numTris) {
					M = 4*_numTris-Math.abs(2*_numTris-1-2*(j+_numTris))-1;
					_spots[i] = new Spot(i,j+_numTris,M,"W","NW",_numberLabel);
				} else if (i<4*_numTris) {
					_spots[i] = new Spot(i,2*_numTris-1,2*_numTris-2*j-1,"NW","NE",_numberLabel);
				} else if (i<5*_numTris) {
					_spots[i] = new Spot(i,2*_numTris-1-j,0,"NE","E",_numberLabel);
				} else {	
					_spots[i] = new Spot(i,_numTris-1-j,0,"E","SE",_numberLabel);
				}
				
				k = (i-j)/_numTris;
				_spots[i].theta = PI*(-0.5+k/3);
				_spotsSprite.addChild(_spots[i]);
			}
		}
		
		/**
		 * Triangle filling functions.
		 */
		 
		//Random fill. 
		public function randomFill(N:int):void {
			
			var totalTris:int = 6*_numTris*_numTris;
			var a:Array = Random.randomCollection(N,totalTris);
		//	trace("length = "+a.length);
			for (var k:int=0; k!=a.length; k++) trace("a["+k+"] = "+a[k]);
			
			var triCount:int = 0;
			var i:int, j:int, L:int;
			for (i=0; i!=_numTris*2; i++) {
				L = 4*_numTris-Math.abs(2*_numTris-1-2*i);
				for (j=0; j!=L; j++) { 
					_triangles[i][j].occupied = a[triCount];
			//		trace("triCount = "+triCount,"a[triCount] = "+a[triCount]);
					triCount++;
				}
			}
		//	dispatchNumberTriangles(triCount);
		} 
			
		//Planned fill.
		public function newFilledTriangles(tris:Array):void {
			var triCount:int = 0;
			var i:int, j:int,L:int,M:int, I:int,J:int;
			M = tris.length;
			for (i=0; i!=M; i++) {
				I = tris[i].I;
				J = tris[i].J;
				if (I>=0 && I<2*_numTris && J>=0 && J<4*_numTris-Math.abs(2*_numTris-1-2*I)) {
					_triangles[I][J].occupied = true;
					triCount++;	
				}
			}
	//		dispatchNumberTriangles(triCount);
		}
		/* 
		private function dispatchNumberTriangles(m:int):void {
			var ev:TriCountEvent = new TriCountEvent(TriCountEvent.TRI_COUNT);
			ev.tris = m;
			dispatchEvent(ev);    
		}
		 */
		/**
		 * Event handlers.
		 */ 
		
		//When a new ray is to be created. Note inefficiency here: calculateRays() is called
		//so that all rays are recalculated.
		private function beginRayHandler(event:BeginRayEvent):void {
			var ray:Ray = new Ray();
			ray.startIndex = event.index;
			ray.startSide = event.side;
			ray.vertices.push(event.triangle);
			_rays.push(ray);
			calculateRays();
		}
		
		private function destroyRayHandler(event:DestroyRayEvent):void {
			var startIndex:int = -1;
			var endIndex:int = -1;
			var side:String;
			
			if (event.startIndex>=0) {
				startIndex = event.startIndex;
				side = event.startSide;
			} else if (event.endIndex>=0) {
				endIndex = event.endIndex;
				side = event.endSide;
			} else return;
			
			//Search rays for the ray to be destroyed.
			var L:int = _rays.length,i:int;
			for (i=0; i!=L; i++) {
				if ( (_rays[i].startIndex==startIndex && _rays[i].startSide==side)
				   ||(_rays[i].endIndex==endIndex && _rays[i].endSide==side) )
				   break;  
			}
			//Get rid of ray it.
			_rays.splice(i,1);
			redoSpots();
		}
		
		//Once a triangle clicked, recalculate rays.
		private function triangleClickHandler(event:MouseEvent):void {
			redoSpots();
		}
		
		private function redoSpots():void {
			clearSpots();
			calculateRays();
		}
		
		/** 
		 * Ray calculation. Need to call this lot when a new triangle is introduced, as triangle changes rays.
		 * The rays are _rays[0], _rays[1],.. need to adjust _rays[i].vertices
		 */ 
		
		//Calculate rays and labelling.
		private function calculateRays():void {
			var i:int,j:int;
			var L:int = _rays.length;
			var triData:TriangleData;
			
			for (i=0; i!=L; i++) {
				//First label the initial spot.
				_spots[_rays[i].startIndex].createLabel(_rays[i].startSide,i,Spot.START);
				
				//Calculate destination triangle.
				if (_rays[i].vertices && _rays[i].vertices[0])
					triData = calculateRay(_rays[i]);
				
				//If null then self ray. Otherwise must label the final spot.
				if (triData != null) {
					var sp:Array = findSpotFromTriangle(triData);
					
					//Fill out data for array.
					_rays[i].endIndex = int(sp[0]);
					_rays[i].endSide = String(sp[1]);
					
					//Label the end spot.
					_spots[_rays[i].endIndex].createLabel(_rays[i].endSide,i,Spot.END);
				} 
				
			}
			
			plotRays();
			
		}
		
		private function findSpotFromTriangle(triData:TriangleData):Array {
			var m:int = 0; //Index of spot.
			var side:String; //left or right.
			
			var I:int = triData.I;
			var J:int = triData.J;
			var d:String = triData.direction;
			
			if (I==0 && J%2==1) {
				m = (J-1)/2; 
				side = (d=="NW") ? "left" : "right";
			} else if ( I<_numTris && J==2*(_numTris+I) ) {
				m = _numTris + I;
				side = (d=="NE") ? "left" : "right";
			} else if ( I>=_numTris && J==6*_numTris-2*I-2 ) {
				m = _numTris+I;
				side = (d=="E") ? "left" : "right";
			} else if ( I==2*_numTris-1 && J%2==1) {
				m = 4*_numTris - (J+1)/2;
				side = (d=="SE") ? "left" : "right";
			} else if ( I>=_numTris && J==0) {
				m = 6*_numTris-I-1;
				side = (d=="SW") ? "left" : "right";
			} else if (I<_numTris && J==0) {
				m = 6*_numTris-I-1;
				side = (d=="W") ? "left" : "right";
			}
			
			return [m,side];
		}
		
		//Given an initial triangle, calculate final triangle in chain.
		private function calculateRay(ray:Ray):TriangleData {
			var triTemp:TriangleData;
			var triData:TriangleData = TriangleData(ray.vertices[0]);
			
			//Reset ray.
			ray.vertices = [triData];
			
			//If blocked already then return null.
			if (_triangles[triData.I][triData.J].occupied) {
				return null; 
			} else {
				//Don't iterate more than 1000 times, and break when null.
				for (var i:int = 0; i!=1000; i++) {
					
					triTemp = findNextTriangle(triData);
					if (triTemp==null) {
						break;
					} else {
						triData = triTemp;
						ray.vertices.push(triData);
					}
					
				//	trace("I = "+triData.I,"J = "+triData.J,"direction = "+triData.direction);
				}
				if (i>=1000) {
					Alert.show('Algorithm failiure','Failed to reach boundary.');
				}
				return triData;
			}
			//If all else fails.
			return null;
			
			//	trace("I = "+triData.I,"J = "+triData.J,"direction = "+triData.direction);
		}
		
		// Returns null if reached an edge of the hexagon. 
		private function findNextTriangle(triData:TriangleData):TriangleData {
			if (triData==null) return null;
			
			var I:int = triData.I;
			var J:int = triData.J;
			var d:String = triData.direction;
			
			var i:int,j:int;
			
			// The number of triangles in this row.
			var L:int = 4*_numTris-Math.abs(2*_numTris-1-2*I);
			
			var outputTri:TriangleData = new TriangleData();
			outputTri.direction = d;
			
			switch (d) {
				case "W":
					if (J==0) return null;
					i = I;
					j = J-1;
					if (_triangles[i][j].occupied) {
						outputTri.I = I;
						outputTri.J = J;
						outputTri.direction = ( (I<_numTris && J%2==0) || (I>=_numTris && J%2==1) ) ? "SE" : "NE";
					} else {
						outputTri.I = i;
						outputTri.J = j;
					}
					break;
				case "E":
					if (J==L-1) return null;
					i = I;
					j = J+1;
					if (_triangles[i][j].occupied) {
						outputTri.I = I;
						outputTri.J = J;
						outputTri.direction = ( (I<_numTris && J%2==0) || (I>=_numTris && J%2==1) ) ? "SW" : "NW";
					} else {
						outputTri.I = i;
						outputTri.J = j;
					}
					break;
				case "NW":
					if ( (I==0 && J%2==1) || (I<_numTris && J==0) ) return null;
					if ( (I<_numTris && J%2==0) || (I>=_numTris && J%2==1) ) {
						i = I;
						j = J-1
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "E";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					} else {
						i = I-1;
						j = (I<_numTris) ? J-1 : (I==_numTris) ? J : J+1;
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "SW";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					}
					break;
				case "NE":
					if ( (I==0 && J%2==1) || (I<_numTris && J==L-1) ) return null;
					if ( (I<_numTris && J%2==0) || (I>=_numTris && J%2==1) ) {
						i = I;
						j = J+1
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "W";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					} else {
						i = I-1;
						j = (I<_numTris) ? J-1 : (I==_numTris) ? J : J+1;
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "SE";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					}
					break;
				case "SW":
					if ( (I==2*_numTris-1 && J%2==1) || (I>=_numTris && J==0) ) return null;
					if ( (I<_numTris && J%2==0) || (I>=_numTris && J%2==1) ) {
						i = I+1;
						j = (I<_numTris-1) ? J+1 : (I==_numTris-1) ? J : J-1;
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "NW";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					} else {	
						i = I;
						j = J-1
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "E";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					}
					break;
				case "SE":
					if ( (I==2*_numTris-1 && J%2==1) || (I>=_numTris && J==L-1) ) return null;
					if ( (I<_numTris && J%2==0) || (I>=_numTris && J%2==1) ) {
						i = I+1;
						j = (I<_numTris-1) ? J+1 : (I==_numTris-1) ? J : J-1;
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "NE";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					} else {	
						i = I;
						j = J+1
						if (_triangles[i][j].occupied) {
							outputTri.I = I;
							outputTri.J = J;
							outputTri.direction = "W";
						} else {
							outputTri.I = i;
							outputTri.J = j;
						}
					}
					break;
			}
			return outputTri;
		}
		
		
		/** 
		 * Instructions from HexWindow to redraw; for example, on full screen.
		 */ 
		
		
		//This gets called when full screen or resize.
		public function redraw(size:Number):void {
			//First recalculate the data to create the hexagon.
			_size = 0.88*size; //Times 0.9 so that circles don't stick out of screen.
			_triSide = _size/(2*_numTris);
			_triHeight = _triSide*ROOT3*0.5;
			
			positionTriangles();
			positionSpots();
			plotRays();
		}
		
		/**
		 * Redraw everything!
		 */ 
		
		private function plotRays():void {
			var g:Graphics = _rayShape.graphics;
			g.clear();
			g.lineStyle(RAY_LINE_THICK,RAY_LINE_COL,1);
			var i:int, j:int, L:int = _rays.length, M:int;
			var a:Array;
			var ray:Ray;
			var triData:TriangleData;
			
			for (i=0; i!=L; i++) {
				ray = _rays[i];
				a = findCoords(ray.vertices[0],false);
				g.moveTo(a[0],a[1]);
				M = ray.vertices.length;
				if (M>1) {
					for (j=1; j!=M; j++) {
						a = findCoords(ray.vertices[j],false);
						g.lineTo(a[0],a[1]);
					}
					//Got to do the last one again.
					a = findCoords(ray.vertices[M-1],true);
					g.lineTo(a[0],a[1]);
				}
			}
			
		}
		
		//Find (x,y) coordinates from triangle data.
		//The variable b is true iff final coordinate.
		private function findCoords(triData:TriangleData,b:Boolean):Array {
			var cx:Number,cy:Number; //Centre of triangle.
			var I:int = triData.I;
			var J:int = triData.J;
			var d:String = triData.direction;
			
			var h:Number = _triSide/ROOT3;
			var k:Number = _triHeight-h;
			var angle:Number;
			
			//Find centre.			
			if (I<_numTris) {
				cx = 0.5*_triSide*(J-I-_numTris); 
				if (J%2 ==0 ) {
					cy = h+_triHeight*(I-_numTris);	
					if (b) angle = (d=="W" || d=="NW") ? 7*PI/6 : (d=="SE" || d=="SW") ? PI/2 : 11*PI/6;
					else angle = (d=="E" || d=="SE") ? 7*PI/6 : (d=="NE" || d=="NW") ? PI/2 : 11*PI/6; 
				} else if (J%2 != 0) {
					cy = _triHeight - h + _triHeight*(I-_numTris);
					if (b) angle = (d=="E" || d=="SE") ? PI/6 : (d=="NE" || d=="NW") ? 3*PI/2 : 5*PI/6;
					else angle = (d=="E" || d=="NE") ? 5*PI/6 : (d=="SE" || d=="SW") ? 3*PI/2 : PI/6;
				}
			} else {
				cx = 0.5*_triSide*(J-(3*_numTris-I)+1); 
				if (J%2 != 0 ) {
					cy = h+_triHeight*(I-_numTris);	
					if (b) angle = (d=="W" || d=="NW") ? 7*PI/6 : (d=="SE" || d=="SW") ? PI/2 : 11*PI/6;
					else angle = (d=="E" || d=="SE") ? 7*PI/6 : (d=="NE" || d=="NW") ? PI/2 : 11*PI/6; 
				} else if (J%2 == 0) {
					cy = _triHeight - h + _triHeight*(I-_numTris);
					if (b) angle = (d=="E" || d=="SE") ? PI/6 : (d=="NE" || d=="NW") ? 3*PI/2 : 5*PI/6;
					else angle = (d=="E" || d=="NE") ? 5*PI/6 : (d=="SE" || d=="SW") ? 3*PI/2 : PI/6;
				}
			}
			//trace("angle = "+angle);
			//Now find side midpoint.
			//return[cx,cy];
		//	return[cx,cy-k];
			return[cx+k*Math.cos(angle),cy+k*Math.sin(angle)];
		}
		
		
		
		
		
		
		
		//Calculate the centres of the triangles.
		private function positionTriangles():void {
			var i:int,j:int,L:int,M:int=_numTris*2;
			
			var h:Number = _triSide/ROOT3;
			
			for (i=0; i!=M; i++) {
				L = _triangles[i].length;
				for (j=0; j!=L; j++) {
					
					// New side length.
					_triangles[i][j].side = _triSide;
					
					// New x y coordinates.
					if (i<_numTris) {
						_triangles[i][j].x = 0.5*_triSide*(j-i-_numTris); 
						if (j%2 ==0 ) {
							_triangles[i][j].y = h+_triHeight*(i-_numTris);	
						} else if (j%2 != 0) {
							_triangles[i][j].y = _triHeight - h + _triHeight*(i-_numTris);
						}
					} else {
						_triangles[i][j].x = 0.5*_triSide*(j-(3*_numTris-i)+1); 
						if (j%2 != 0 ) {
							_triangles[i][j].y = h+_triHeight*(i-_numTris);	
						} else if (j%2 == 0) {
							_triangles[i][j].y = _triHeight - h + +_triHeight*(i-_numTris);
						}
					}
					// Redraw it.
					_triangles[i][j].redraw();
				}
			}
			
		}
		
		//Calculate the centres of the triangles.
		private function positionSpots():void {
			var i:int,j:int,k:int,M:int=_numTris*6;
	
			var p:Number = _triSide*ROOT3/6;
			var q:Number = _triSide/ROOT3;
			
			/** First calculate radius of spot:
			 *  radius = prop*_triSide
			 *  In fact use (prop-0.01)*_triSide to avoid going over hexagon.
			 *  Nonlinear formula for prop depending on _numSides so that
			 *  don't get too small circles.
			 */ 
			
			//ROOT3/6 = 0.2886 
			var prop:Number = (_numTris==2) ? 0.24 :
			                  (_numTris==3) ? 0.29 :
			                  (_numTris==4) ? 0.34 :
			                  (_numTris==5) ? 0.39 :
			                  (_numTris==6) ? 0.44 : 0.44;
/* 			                   

			r = _triSide*SPOT_RAD_PROP;
			if (_numTris==2) r *= 0.8;
			
*/			
			 
			var r:Number = _triSide*prop; 			
 			var rx:Number = r*ROOT3*0.5;
			var ry:Number = r*0.5;	
			
			//Make each spot just touch triangle
			for (i=0; i!=M; i++) {
				_spots[i].radius = _triSide*(prop-0.01);
				j = i%_numTris;
				if (i<_numTris) {
					_spots[i].x = (-0.5*(_numTris-1)+j)*_triSide; 
					_spots[i].y = -_numTris*_triHeight-r;
				} else if (i<2*_numTris) {
					_spots[i].x = 0.5*(_numTris+0.5+j)*_triSide+rx; 
					_spots[i].y = (j-_numTris+0.5)*_triHeight-ry;
				} else if (i<3*_numTris) {
					_spots[i].x = (_numTris-0.5*j-0.25)*_triSide+rx; 
					_spots[i].y = (j+0.5)*_triHeight+ry;
				} else if (i<4*_numTris) {
					_spots[i].x = (0.5*(_numTris-1)-j)*_triSide; 
					_spots[i].y = _numTris*_triHeight+r;
				} else if (i<5*_numTris) {
					_spots[i].x = -0.5*(_numTris+0.5+j)*_triSide-rx;
					_spots[i].y = (_numTris-j-0.5)*_triHeight+ry;
				} else {	
					_spots[i].x = (0.5*j+0.25-_numTris)*_triSide-rx; 
					_spots[i].y = (-j-0.5)*_triHeight-ry;
				}
				
				/** Could consider adjusting spot radii depending
				 * on _numTris. For example, make it bigger as 
				 * _numTris gets larger. For now I'll just shrink a 
				 * little when _numTris=2 else not fit on screen.
				 */ 
				
				
				_spots[i].redraw();
			}
			
		}
		
		/**
		 * Clear functions.
		 */ 
		
		public function clearAll():void {
			clearTriangles();
			clearSpots();
			clearRays();
		}
		
		private function clearTriangles():void {
			var i:int, j:int,L:int;
			for (i=0; i!=_numTris*2; i++) {
				L = 4*_numTris-Math.abs(2*_numTris-1-2*i);
				for (j=0; j!=L; j++) _triangles[i][j].clear();
			}
		}
		
		private function clearSpots():void {
			var L:int = _spots.length;
	//		trace("L = "+L);
			for (var i:int=0; i!=L; i++) {
	//			trace("_spots[i] = "+_spots[i]);
				_spots[i].clearLabels();
			}
		}
		
		private function clearRays():void {
			_rays = [];
			_rayShape.graphics.clear();
		}
		 
	}
}
