function Point(x, y) 
{
  this.x = x;
  this.y = y;
}

Point.prototype.max = function(aPoint)
{
  return this.magnitude > aPoint.magnitude ? this : aPoint;
};

Point.prototype.magnitude = function()
{
  return Math.sqrt(this.x * this.x + this.y * this.y);
};


function Turtle(screen)
{
  this.x = 0;
  this.y = 0;
  this.h = 0;
  this.penIsDown = true;
  this.turtleIsVisible = false;
  this.color = '#000000';
  this.screen = screen;
}

Turtle.prototype.setSprite = function(sprite)
{
  this.sprite = sprite;
  this.turtleIsVisible = true;
  this.screen.initSprite(this.sprite);
};

Turtle.prototype.clearscreen = function()
{
  this.screen.clear();
  if (this.sprite) this.sprite.home();
  this.x = 0;
  this.y = 0;
  this.h = 0;
}

Turtle.prototype.forward = function(n)
{
  var theta = this.h * Math.PI / 180
  var x = this.x + Math.sin(theta) * n;
  var y = this.y + Math.cos(theta) * n;
  
  if (this.sprite) 
  {
    this.sprite.goTo(x,y);
  }
  
  if (this.penIsDown)
  {
    this.screen.color(this.color)
    this.screen.line(this.x, this.y, x, y);
  }
    
  this.x = x;
  this.y = y;
}

Turtle.prototype.right = function(n)
{
  this.h = (this.h + n) % 360;
  if (this.sprite && this.turtleIsVisible) this.sprite.rotateTo(this.h);
}

Turtle.prototype.penup = function()
{
  this.penIsDown = false;
}

Turtle.prototype.pendown = function()
{
  this.penIsDown = true;
}

Turtle.prototype.hide = function()
{
  this.sprite.clear();
  this.turtleIsVisible = false;
};

Turtle.prototype.show = function()
{
  this.sprite.goTo(this.x, this.y);
  this.sprite.rotateTo(this.h);
  this.turtleIsVisible = true;
};

Turtle.prototype.addPrimitives = function(namespace)
{
  var self = this;
  
  namespace.clearscreen = function(k)
  {
    self.clearscreen();
    k() 
  }

  namespace.forward = function(k, n)
  {
    self.forward(n);
    this.delay(k, n);
  }

  namespace.right = function(k, n)
  {
    self.right(n);
    k()
  }
  
  namespace.penup = function(k)
  {
    self.penup();
    k()
  }
  
  namespace.pendown = function(k)
  {
    self.pendown();
    k()
  }

  namespace.hideturtle = function(k)
  {
    self.hide();
    k();
  }

  namespace.showturtle = function(k)
  {
    self.show();
    k();
  }

  namespace.color = function(k, color)
  {
    self.color = color;
    k();
  }

  namespace.rgba = function(k, r, g, b, a)
  {
    self.color = 'rgba(' + [r,g,b,a].join() + ')';
    k();
  }
  
  namespace.xpos = function(k)
  {
    k(self.x)
  }

  namespace.ypos = function(k)
  {
    k(self.y)
  }

  namespace.heading = function(k)
  {
    k(self.h)
  }

  // short forms
  namespace.cs = namespace.clearscreen;
  namespace.fd = namespace.forward;
  namespace.rt = namespace.right;
  namespace.pu = namespace.penup;
  namespace.pd = namespace.pendown;
  namespace.ht = namespace.hideturtle;
  namespace.st = namespace.showturtle;
    
}


function Screen(canvas)
{
  if (canvas) 
  {
    this.canvas = canvas;
    this.initialize();
  }
}

Screen.prototype.initialize = function()
{
  this.ctx = this.canvas.getContext("2d");
  this.ctx.lineCap = 'round';
  this.ctx.lineJoin = 'round';
	this.ctx.translate(this.width()/2, this.height()/2);
	this.ctx.save();
};

Screen.prototype.clear = function()
{
	this.ctx.clearRect(
		-this.width()/2, 
		-this.height()/2, 
		this.width(),
		this.height()
	);
}

Screen.prototype.color = function(color)
{
	this.ctx.strokeStyle = color;
}

Screen.prototype.line = function(ox, oy, dx, dy)
{
  this.ctx.beginPath();
  this.ctx.moveTo(ox, -oy);
  this.ctx.lineTo(dx, -dy);
  this.ctx.stroke();
}

Screen.prototype.width = function()
{
  return this.canvas.getAttribute('width');
};

Screen.prototype.height = function()
{
  return this.canvas.getAttribute('height');
};

Screen.prototype.size = function()
{
  var w = this.width();
  var h = this.height();
  return w < h ? w : h;
};

Screen.prototype.initSprite = function(sprite)
{
  sprite.setScreen(this.canvas)
};


function Sprite(node)
{
  this.canvas = document.createElement('canvas');
  this.canvas.style.position = 'absolute';
  this.canvas.setAttribute('width', 32);
  this.canvas.setAttribute('height', 32);
  
  if (node && node.tagName == 'CANVAS')
    this.renderer = new DrawingRenderer(this, node);
  else if (node && node.tagName == 'IMG')
    this.renderer = new ImageRenderer(this, node);
  else if (node) 
    this.renderer = new LogoRenderer(this, node);
  else 
    this.renderer = new JavascriptRenderer(this, node);
}

Sprite.prototype = new Screen;

Sprite.prototype.setScreen = function(node)
{
  this.backdrop = node;
  this.backdrop.parentNode.insertBefore(this.canvas, this.backdrop);
  this.initialize();
  this.home();
};

Sprite.prototype.home = function()
{
  this.goTo(0,0);
  this.rotateTo(0);
};

Sprite.prototype.clear = function()
{
  var w = this.canvas.getAttribute('width');
  var h = this.canvas.getAttribute('height');
  var x = w / 2 * -1;
  var y = h / 2 * -1;
  this.ctx.clearRect(x, y, w, h);
};

Sprite.prototype.goTo = function(x, y)
{
  var cw = this.backdrop.getAttribute('width');
  var ch = this.backdrop.getAttribute('height');
  var sw = this.width()
  var sh = this.height();

  var ax = this.backdrop.offsetLeft + (cw - sw)/2 ;
  var ay = this.backdrop.offsetTop + (ch - sh)/2;

  this.canvas.style.left = (ax + x) + 'px';
  this.canvas.style.top = (ay - y) + 'px';
};

Sprite.prototype.rotateTo = function(heading)
{
  var radians = heading * Math.PI / 180;
  this.ctx.restore();
  this.clear();
  this.ctx.save();
  this.ctx.rotate(radians);
  this.draw();
};

Sprite.prototype.draw = function()
{
  this.renderer.renderOn(this.ctx);
};

DrawingRenderer = function(sprite, slate)
{
  this.screen = sprite;
  this.drawing = new Drawing();
}

DrawingRenderer.prototype.renderOn = function(ctx)
{
  this.drawing.drawOn(this.screen, this.factor);
};

DrawingRenderer.prototype.setDrawing = function(aDrawing)
{
  this.drawing = aDrawing;
  this.factor = aDrawing.scaleFactorToFit(this.screen.size());
  this.renderOn()
};

ImageRenderer = function(sprite, image) 
{
  this.image = image;
}

ImageRenderer.prototype.renderOn = function(ctx)
{
  var x = 0 - this.image.width / 2;
  var y = 0 - this.image.height / 2;
  ctx.drawImage(this.image, x, y);
};

LogoRenderer = function(sprite, node) {
  this.code = node;
  this.code.value = this.code.firstChild.data;
  this.turtle = new Turtle(sprite);
  this.evaluator = new Evaluator(this.turtle);
  this.evaluator.scheduler.speed = 0;
}

LogoRenderer.prototype.renderOn = function(ctx)
{
  this.turtle.clearscreen();
  this.evaluator.eval(this.code.value)
};

JavascriptRenderer = function(canvas) {}

JavascriptRenderer.prototype.renderOn = function(ctx)
{
  var style = ctx.fillStyle;
  ctx.fillStyle = 'rgba(100, 100, 192, 0.4)';
  ctx.beginPath();
  ctx.moveTo(0, -8);
  ctx.lineTo(8, 8);
  ctx.quadraticCurveTo(0,-1,-8,8);
  ctx.closePath();
  ctx.fill();
  ctx.fillStyle = style;
};


function Slate(node) 
{
	var self = this;
	this.canvas = node;
  this.initialize();
  this.ctx.lineWidth = 8;
  this.ctx.lineCap = 'round';
  this.ctx.lineJoin = 'round';
  this.clear();
	this.drawing = new Drawing();
	this.onMouseDown = function(evt) {self.mousedown(evt)};
  this.canvas.addEventListener('mousedown', this.onMouseDown, false);
  node.drawing = this.drawing;
}

Slate.prototype = new Screen;

Slate.prototype.clear = function()
{
  var style;
  Screen.prototype.clear.call(this)
  style = this.ctx.strokeStyle;
  this.ctx.strokeStyle = '#eeeeee';
  this.line(0, 20, 0, -20);
  this.line(-20, 0, 20, 0);
  this.ctx.strokeStyle = style;
};

Slate.prototype.redraw = function()
{
  this.drawing.drawOn(this);
};

Slate.prototype.reset = function()
{
  this.clear();
  this.drawing.reset();
};

Slate.prototype.mousedown = function(event)
{
	new Gesture(event, this, this.drawing);
};

function Gesture(event, slate, drawing) {
  var self = this;
	this.slate = slate;
	this.drawing = drawing;
  this.points = new Array;
  
  this.initConversion(event);
	this.onMouseMove = function(evt) {self.mousemove(evt)};
	this.onMouseUp = function(evt) {self.mouseup(evt)};
  window.addEventListener('mousemove', this.onMouseMove, true);
  window.addEventListener('mouseup', this.onMouseUp, true);
  
  this.points.push(this.point(event));
}

Gesture.prototype.initConversion = function(down)
{
  var offsetX = down.clientX - down.layerX + down.target.offsetLeft;
  var offsetY = down.clientY - down.layerY + down.target.offsetTop;
  var translateX = down.target.getAttribute('width')/2;
  var translateY = down.target.getAttribute('height')/2;

  this.point = function(move)
  {
    var x = move.clientX - offsetX - translateX;
    var y = translateY - (move.clientY - offsetY)
    return new Point(x, y);
  }
};

Gesture.prototype.mousemove = function(event)
{
	var previous = this.points[this.points.length-1];
  var point = this.point(event);
  this.slate.line(previous.x, previous.y, point.x, point.y);
  this.points.push(point);
};

Gesture.prototype.mouseup = function(event)
{
  this.mousemove(event);
  window.removeEventListener('mousemove', this.onMouseMove, true);
  window.removeEventListener('mouseup', this.onMouseUp, true);
	this.drawing.addStroke(new Stroke(this.points));
};

function Drawing()
{
	this.strokes = new Array();
}

Drawing.prototype.reset = function()
{
  this.strokes = new Array();
};

Drawing.prototype.drawOn = function(screen, factor)
{
  screen.ctx.save()
  if (factor) 
  {
    screen.ctx.scale(factor, factor);
    screen.ctx.lineWidth = 1/factor;
  }
	for (var i=0; i<this.strokes.length; i++)
    this.strokes[i].drawOn(screen);
  screen.ctx.restore();
};

Drawing.prototype.addStroke = function(stroke)
{
	this.strokes.push(stroke);
};

Drawing.prototype.scaleFactorToFit = function(width, height)
{
  var size = this.size();
  if (width > height) 
    return height / size;
  else 
    return width / size;
};

Drawing.prototype.size = function()
{
  var max = new Point(0,0);
  for (var i=0; i<this.strokes.length; i++)
  {
    size = this.strokes[i].size();
    max = max > size ? max : size;
  }
  return max;
};

function Stroke(points)
{
	this.points = points;
}

Stroke.prototype.drawOn = function(aScreen)
{
  for (var i=1; i<this.points.length; i++)
	{
		var origin = this.points[i-1];
		var destination = this.points[i];
    aScreen.line(origin.x, origin.y, destination.x, destination.y);
	}
};

Stroke.prototype.size = function()
{
  var current;
  var max = this.points[0].magnitude;
  for (var i=1; i<this.points.length; i++)
  {
    current = this.points[i].magnitude();
    max = max > current ? max : current;
  }
    
  return max * 2;
};

