﻿import SetValue;
class Complex {
	//
	// A Complex numbers implementation.
	// - Also implements Boolean, Set and symbol values. 
	//
	// TODO: Refactor this into a general purpose Value class plus classes representing
	// the pure data types. 
	//
	var x:Number;
	var y:Number;
	private var _setValue:SetValue = null;	// needed so we can evaluate Sets too
	private var _symbolValue = null; // and now strings too
	// Booleans have true == 1, false == 0
	
	static var INFINITY:Complex;	// initialised elsewhere before use!
	
	static var inverse:Object = {
		negate:"negate",
		reciprocal:"reciprocal",
		conjugate:"conjugate",
		abs:"abs",
		exp:"ln",
		ln:"exp",
		sqr:"sqrt",
		sqrt:"sqr",
		cos:"acos",
		acos:"cos",
		sin:"asin",
		asin:"sin",
		tan:"atan",
		atan:"tan"	
	}
	
	function Complex(xr:Object, ytheta:Number, polar:Boolean) {
		//setValue = new SetValue([this]);	// implements a type cast to Set
		if(polar) {
			x = Number(xr)*Math.cos(ytheta);
			y = Number(xr)*Math.sin(ytheta);
		}
		else if(xr.valueOf() == Number(xr)) {
			this.x = Number(xr);
			this.y = ytheta == null ? 0 : ytheta;
		}
		else if(xr instanceof SetValue) {
			_setValue = SetValue(xr);
			//trace("Complex:48: Creating set value: " + _setValue);
		}
		else if(xr.toString() == xr.valueOf()) {
			_symbolValue = xr.toString();
			//trace("Complex:49: Creating string value: " + _setValue);
		}
		toString = displayString;
	}
	
	public static function classInit():Void {
		INFINITY = new Complex(Number.POSITIVE_INFINITY);
	}
	
	public function hasFunction(s:String):Boolean {
		return this[s] instanceof Function;
	}
	
	//
	// get setValue() provides a dynamic cast to setValue
	//
	function get setValue():SetValue {
		// if this is a set, return its value
		if(_setValue != null) {
			return _setValue;
		}
		// else return a set containing just this value
		var members:Object = {};
		members[toString()]=this;
		return new SetValue(members);
	}
	
	function isSet():Boolean {
		return _setValue != null;
	}
	
	function isSymbol():Boolean {
		return _symbolValue != null;
	}
	//
	// Relational expressions:
	//
	function boolEquals(c:Complex):Boolean {
		return (x==c.x && y == c.y);
	}
	
	function equals(c:Complex):Complex {
		return boolEquals(c) ? new Complex(1) : new Complex(0);
	}
	
	function notEquals(c:Complex):Complex {
		return boolEquals(c) ? new Complex(0) : new Complex(1);
	}
	
	function logicalNot():Complex {
		return (x==1 && y==0) ? new Complex(1) : new Complex(0);
	}
	
	//
	// Relational expressions which require real operands
	//
	function greaterThan(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x > c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function greaterEquals(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x >= c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function lessThan(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x < c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function lessEquals(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x <= c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function times(z:Object):Complex {
		if(zfinite(z)) {
			if(z.valueOf() == Number(z).valueOf()) {
				return new Complex(x*z, y*z);
			}
			if(z instanceof Complex) {
				return new Complex(x*z.x-y*z.y, x*z.y+y*z.x);
			}
		}
		else {
			return INFINITY; // ignore INFINITY*0 case
		}
	}
	
	function over(z:Object):Complex {
		if(zfinite(z)) {
			if(z.valueOf() == Number(z).valueOf()) {
				return z == 0 ? INFINITY: new Complex(x/z, y/z);
			}
			var d:Number = z.x*z.x + z.y*z.y;
			return d == 0 ? INFINITY: new Complex((x*z.x+y*z.y)/d, (-x*z.y+y*z.x)/d);
		}
		else {
			if(finite()) {
				return new Complex(0);
			}
			else {
				return INFINITY;
			}
		}
	}
	
	function plus(z:Object):Complex {
		if(!zfinite(z)) return INFINITY;
		if(z.valueOf() == Number(z).valueOf()) {
			return new Complex(x+z, y);
		}
		return new Complex(x+z.x, y+z.y);
	}
	
	function minus(z:Object):Complex {
		if(!zfinite(z)) return INFINITY;
		if(z.valueOf() == Number(z).valueOf()) {
			return new Complex(x-z, y);
		}
		return new Complex(x-z.x, y-z.y);
	}
	
	function negate():Complex {
		if(!finite()) return INFINITY;
		return new Complex(-x,-y);
	}
	
	function conjugate():Complex {
		if(!finite()) return INFINITY;
		return new Complex(x, -y);
	}
	
	function reciprocal():Complex {
		if(finite()) {
			var d = x*x - y*y;
			return d == 0 ? INFINITY: (new Complex(1)).over(this);
		}
		return new Complex(0);
	}
	
	function abs():Complex {
		if(!finite()) return INFINITY;
		return new Complex(length, 0);
	}
	
	function arg():Complex {
		if(!finite()) return new Complex(0);
		return new Complex(angle, 0);
	}
	
	function exp():Complex {
		if(!finite()) return INFINITY;
		var ex = Math.exp(x);
		return new Complex(ex*Math.cos(y), ex*Math.sin(y));
	}
	
	function sin():Complex {
		if(!finite()) return INFINITY;
		var eiz:Complex = (new Complex(-y,x)).exp();
		var emiz:Complex = (new Complex(y,-x)).exp();
		return eiz.minus(emiz).over(new Complex(0,2));
	}
	
	function asin():Complex {
		if(!finite()) return INFINITY;
		var i:Complex = new Complex(0,1);
		var w:Complex = ((new Complex(1)).minus(times(this))).sqrt().plus(times(i));
		return w.ln().times(i).negate();
	}
	
	function cos():Complex {
		if(!finite()) return INFINITY;
		var eiz = (new Complex(-y,x)).exp();
		var emiz = (new Complex(y,-x)).exp();
		return eiz.plus(emiz).over(2);
	}
	
	function acos():Complex {
		//pi/2+i*ln(i*z+sqrt(1-z^2))
		if(!finite()) return INFINITY;
		var i:Complex = new Complex(0,1);
		var w:Complex = ((new Complex(1)).minus(times(this))).sqrt().times(i).plus(times(i));
		return w.ln().times(i).plus(new Complex(Math.PI/2));
	}
	
	function tan():Complex {
		if(!finite()) return INFINITY;
		return sin().over(cos());
	}
		
	function atan():Complex {
		//(ln(1-iz)-ln(1+iz))*i/2
		if(!finite()) return new Complex(Math.PI/2);
		var i:Complex = new Complex(0,1);
		var one:Complex = new Complex(1);
		var iz = times(i);
		var ln1 = one.minus(iz).ln();
		var ln2 = one.plus(iz).ln();
		return ln1.minus(ln2).times(i).over(2);
	}
	
	function sinh():Complex {
		if(!finite()) return INFINITY;
		var ez:Complex = exp();
		var emz:Complex = (new Complex(-x,-y)).exp();
		//trace(ez.minus(emz).over(2));
		return ez.minus(emz).over(2);
	}
	
	function asinh():Complex {
		if(!finite()) return INFINITY;
		// ln(z)+sqrt(1+z^2)
		return ln().plus((sqr().plus(new Complex(1))).sqrt());
	}
	
	function cosh():Complex {
		if(!finite()) return INFINITY;
		var ez:Complex = exp();
		var emz:Complex = (new Complex(-x,-y)).exp();
		//trace(ez.plus(emz).over(2));
		return ez.plus(emz).over(2);
	}
	
	function acosh():Complex {
		if(!finite()) return INFINITY;
		// ln(z)+sqrt(1+z^2)
		return ln().plus((sqr().minus(new Complex(1))).sqrt());
	}
	
	function tanh():Complex {
		if(!finite()) return INFINITY;
		return sinh().over(cosh());
	}

	function atanh():Complex {
		if(!finite()) return INFINITY;
		var one:Complex = new Complex(1);
		plus(one).ln().minus(one.minus(ln())).over(2);
	}
	
	function ln():Complex {
		if(!finite()) return INFINITY;
		var len = length;
		return negligible(len) ? INFINITY: new Complex(Math.log(len), angle); 
	}
	
	function sqrt():Complex {
		if(!finite()) return INFINITY;
		return ln().times(0.5).exp();
	}
	
	function sqr():Complex {
		if(!finite()) return INFINITY;
		return times(this);
	}
	
	function pow(z:Complex):Complex {
		if(!finite()) return INFINITY;
		var log:Complex = ln();
		return log.finite() ? ln().times(z).exp() : (z.x > 0) ? new Complex(0) : INFINITY;
	}
	
	function round(z:Complex):Complex {
		if(!finite()) return INFINITY;
		return new Complex(Math.round(z.x), Math.round(z.y));
	}
	
	function get length():Number {
		if(!finite()) return Number.POSITIVE_INFINITY;
		return Math.sqrt(x*x+y*y);
	}
	
	function get angle():Number {
		if(!finite()) return 0;
		return Math.atan2(y,x);
	}
	
	function polar():Object {
		return {r:length, theta:angle};
	}
	
	/* User functions
	function T():Complex {
		return new Complex(x+1, y);
	}
	function t():Complex {
		return new Complex(x-1, y);
	}
	function R():Complex {
		return new Complex(-x, y);
	}
	function r():Complex {
		return new Complex(-x, y);
	}
	function S():Complex {
		return new Complex(-y, x);
	}
	function s():Complex {
		return new Complex(y, -x);
	}
	*/

/*
 * Move this to a conversion class to avoid dependency on Matrix
 *
	function matrix():Matrix {
		return new Matrix(x, -y, y, x);
	}
	
	function fromMatrix(m:Matrix):Complex {
		if(m.tx != 0 || m.ty != 0 || m.a != m.d || m.c != -m.b) {
			return null;
		}
		return new Complex(m.a, m.c);
	}
 *
 */
 
 	function negligible(t:Number):Boolean {
		return Math.abs(t) < 1e-14;
	}
	
	function finite():Boolean {
		return isFinite(x) && isFinite(y);
	}
	
	function zfinite(z:Object):Boolean {
		if(z instanceof Complex) {
			return isFinite(x) && isFinite(y) && isFinite(z.x) && isFinite(z.y);
		}
		else {
			return isFinite(z.valueOf());
		}
	}
	
	function nan():Boolean {
		return isNaN(x) || isNaN(y);
	}
	
	function specialString():String {
		if(_setValue != null) {
			return ""+_setValue;
		}
		if(_symbolValue != null) {
			return ""+_symbolValue;
		}
		if(!finite()) {
			return "Infinity";
		}
		if(nan()) {
			return "NaN";
		}
		if(negligible(y) && Math.abs(x) < 4 && Math.abs(x) > 2) {
			if(negligible(x-Math.E)) 
				return "e";
			if(negligible(x+Math.E)) {
				return "-e";
			}
			if(negligible(x-Math.PI)) 
				return "pi";
			if(negligible(x+Math.PI)) 
				return "-pi";
		}
		return null
	}
	
	function fullString():String {
		var s:String;
		if((s=specialString())!=null) {
			return s;
		}

		if(negligible(y)) {			
			return ""+x;
		}
		var im:String;
		if(y<0) {
			im = negligible(y+1) ? "-i" : ""+y+"*i";
			return negligible(x) ? im : "" + x +im;
		}
		else {
			im = negligible(y-1) ? "i" : ""+y+"*i";
			return negligible(x) ? im : "" + x + "+" +im;
		}
	}
	
	function displayString():String {
		//
		// Display 3 sig fig only
		//
		var s:String;
		if((s=specialString())!=null) {
			return s;
		}

		var z:Complex = new Complex(toDP(x,2),toDP(y,2));
		return z.fullString();
	}
	
	function toDP(t:Number, dps:Number):Number {
		var whole:Number = Math.floor(t);
		var pten:Number = Math.pow(10,dps);
		var frac:Number = (t-whole)*pten;
		return whole + Math.floor(frac)/pten
	}
	
	function toSigFig(t:Number, sigfigs:Number):Number {
		if(t==0) return t;
		var sign:Number=t/Math.abs(t);
		t = Math.abs(t);
		for(var k=0; t >= 10; k++) {
			t /= 10;
		}
		if(k==0) for(k=0; t < 1; k--) {
			t *= 10;
		}
		var n = (Math.pow(10,sigfigs-1));
		t = Math.round(t*n)/n;
		t *= (sign*Math.pow(10,k));
		return t;
	}
	
	// ------------------------------------------------------------------------
	
	//
	// Set evaluation functions (delegated)
	//
	function union(z:Complex):Complex {
		var sv:SetValue = setValue.union(z.setValue);
		//trace("creating new set: " + sv);
		return new Complex(sv);
	}
	
	// ------------------------------------------------------------------------
	static function test():Void {
		var a = new Complex(1,1);
		var b = new Complex(Math.sqrt(2), Math.PI/4, true);
		trace("sin(1,1) = "+ a.sin() + " sin(root2, pi/4) = " + b.sin());
		trace("|<3,4>|"+(new Complex(3,4)).length);
		trace("<1,1>*<1,1>="+a.times(b));
	}
}