﻿import SetValue;
import PrimitiveSet;
import Complex;

class MValue {
	//
	// This class is a wrapper for the various mathematical values.
	// It's the common return value for evaluated expressions
	//
	public static var COMPLEX:Number = 0;
	public static var REAL:Number = 1;
	public static var LIST:Number = 2;
	public static var PRIMITIVESET:Number = 3;
	public static var IMAGESET:Number = 4;
	public static var SYMBOL:Number = 5;

	private var _type:Number = 0;
	private var _value:Object = null;
	// Booleans have true == 1, false == 0
	
	function MValue(xr:Object, ytheta:Number, polar:Boolean) {
		if(polar) {
			_type = COMPLEX;
			x = Number(xr)*Math.cos(ytheta);
			y = Number(xr)*Math.sin(ytheta);
		}
		else if(xr.valueOf() == Number(xr)) {
			_type = REAL;
			x = Number(xr);
			if(ytheta != null) {
				_type = COMPLEX;
				y = ytheta;
			}
		}
		else if(xr instanceof Complex) {
			_type = COMPLEX;
			_value = xr;
		}
		else if(xr instanceof SetValue) {
			_type = LIST;
			_value = SetValue(xr);
			//trace("MValue:42: Creating set value: " + _setValue);
		}
		else if(xr.toString() == xr.valueOf()) {
			var s:String = xr.toString();
			if(PrimitiveSet[s] != null) {
				_type = PRIMITIVESET;
				_value = PrimitiveSet[s];
			}
			else {
				_type = SYMBOL;
				_value = xr.toString();
				//trace("MValue:53: Creating string value: " + _setValue);
			}
		}
	}
	
	function get type():String {
		switch(_type) {
			case COMPLEX: return "Complex";
			case REAL: return "Real";
			case LIST: return "List";
			case PRIMITIVESET: return PrimitiveSet(_value).toString();
			case IMAGESET: return "ImageSet";
			case SYMBOL: return "Symbol";
		}
	}
	
	function get x():Number {
		if(_type == COMPLEX || _type == REAL)
			return _value.x;
		throw new Error(type + ".x is invalid");
	}

	function put x(xx:Number):Void {
		if(_type == COMPLEX || type == REAL) {
			_value.x = xx;
			return;
		}
		throw new Error(type + ".x is invalid");		
	}

	function get y():Number {
		if(_type == COMPLEX) {
			return _value.y;
		}
		throw new Error(type + ".y is invalid");
	}

	function put y(yy:Number):Void {
		if(_type == COMPLEX) {
			_value.y = yy;
			return;
		}
		throw new Error(type + ".x is invalid");		
	}
//
	// get setValue() provides a dynamic cast to setValue
	//
	function get setValue():SetValue {
		// if this is a set, return its value
		if(_setValue != null) {
			return _setValue;
		}
		// else return a set containing just this value
		var members:Object = {};
		members[toString()]=this;
		return new SetValue(members);
	}
	
	function isSet():Boolean {
		return _setValue != null;
	}
	
	function isSymbol():Boolean {
		return _symbolValue != null;
	}
	//
	// Relational expressions:
	//
	function boolEquals(c:Complex):Boolean {
		return (x==c.x && y == c.y);
	}
	
	function equals(c:Complex):Complex {
		return boolEquals(c) ? new Complex(1) : new Complex(0);
	}
	
	function notEquals(c:Complex):Complex {
		return boolEquals(c) ? new Complex(0) : new Complex(1);
	}
	
	//
	// Relational expressions which require real operands
	//
	function greaterThan(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x > c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function greaterEquals(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x >= c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function lessThan(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x < c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function lessEquals(c:Complex):Complex {
		if(y != 0 || c.y != 0) {
			throw new Error(""+this + " and " + c + " cannot be compared");
		}
		if(x <= c.x) {
			return new Complex(1);
		}
		else {
			return new Complex(0);
		}
	}
	
	function times(z:Object):Complex {
		if(z.valueOf() == Number(z).valueOf()) {
//		if(z instanceof Number) {
			return new Complex(x*z, y*z);
		}
		if(z instanceof Complex) {
			return new Complex(x*z.x-y*z.y, x*z.y+y*z.x);
		}
	}
	
	function over(z:Object):Complex {
		if(z.valueOf() == Number(z).valueOf()) {
//		if(z instanceof Number) {				
			return z == 0 ? new Complex(Number.POSITIVE_INFINITY): new Complex(x/z, y/z);
		}
		var d:Number = z.x*z.x + z.y*z.y;
		return d == 0 ? new Complex(Number.POSITIVE_INFINITY): new Complex((x*z.x+y*z.y)/d, (-x*z.y+y*z.x)/d);
	}
	
	function plus(z:Object):Complex {
		if(z.valueOf() == Number(z).valueOf()) {
//		if(z instanceof Number) {
			return new Complex(x+z, y);
		}
		return new Complex(x+z.x, y+z.y);
	}
	
	function minus(z:Object):Complex {
		if(z.valueOf() == Number(z).valueOf()) {
//		if(z instanceof Number) {
			return new Complex(x-z, y);
		}
		return new Complex(x-z.x, y-z.y);
	}
	
	function negate():Complex {
		return new Complex(-x,-y);
	}
	
	function conjugate():Complex {
		return new Complex(x, -y);
	}
	
	function reciprocal():Complex {
		var d = x*x - y*y;
		return d == 0 ? new Complex(Number.POSITIVE_INFINITY): (new Complex(1)).over(this);
	}
	
	function abs():Complex {
		return new Complex(length, 0);
	}
	
	function arg():Complex {
		return new Complex(angle, 0);
	}
	
	function exp():Complex {
		var ex = Math.exp(x);
		return new Complex(ex*Math.cos(y), ex*Math.sin(y));
	}
	
	function cos():Complex {
		var eiz = (new Complex(-y,x)).exp();
		var emiz = (new Complex(y,-x)).exp();
		return eiz.plus(emiz).over(2);
	}
	
	function acos():Complex {
		//pi/2+i*ln(i*z+sqrt(1-z^2))
		var i:Complex = new Complex(0,1);
		var w:Complex = ((new Complex(1)).minus(times(this))).sqrt().times(i).plus(times(i));
		return w.ln().times(i).plus(new Complex(Math.PI/2));
	}
	
	function sin():Complex {
		var eiz = (new Complex(-y,x)).exp();
		var emiz = (new Complex(y,-x)).exp();
		return eiz.minus(emiz).over(new Complex(0,2));
	}
	
	function asin():Complex {
		//-i*ln(i*z+sqrt(1-z^2))
		var i:Complex = new Complex(0,1);
		var w:Complex = ((new Complex(1)).minus(times(this))).sqrt().plus(times(i));
		return w.ln().times(i).negate();
	}
	
	function tan():Complex {
		return sin().over(cos());
	}
		
	function atan():Complex {
		//(ln(1-iz)-ln(1+iz))*i/2
		var i:Complex = new Complex(0,1);
		var one:Complex = new Complex(1);
		var iz = times(i);
		var ln1 = one.minus(iz).ln();
		var ln2 = one.plus(iz).ln();
		return ln1.minus(ln2).times(i).over(2);
	}
	
	function ln():Complex {
		var len = length;
		return len < 1e-15 ? new Complex(Number.POSITIVE_INFINITY): new Complex(Math.log(len), angle); 
	}
	
	function sqrt():Complex {
		return ln().times(0.5).exp();
	}
	
	function sqr():Complex {
		return times(this);
	}
	
	function pow(z:Complex):Complex {
		return ln().times(z).exp();
	}
	
	function get length():Number {
		return Math.sqrt(x*x+y*y);
	}
	
	function get angle():Number {
		return Math.atan2(y,x);
	}
	
	function polar():Object {
		return {r:length, theta:angle};
	}

/*
 * Move to a conversion class so we don't have a dependency on Matrix
 *
	function matrix():Matrix {
		return new Matrix(x, -y, y, x);
	}
	
	function fromMatrix(m:Matrix):Complex {
		if(m.tx != 0 || m.ty != 0 || m.a != m.d || m.c != -m.b) {
			return null;
		}
		return new Complex(m.a, m.c);
	}
 *
 */
	function finite():Boolean {
		return isFinite(x) && isFinite(y);   // need something else for sets
	}
	
	function nan():Boolean {
		return isNaN(x) || isNaN(y);
	}
	
	function specialString():String {
		if(_setValue != null) {
			return ""+_setValue;
		}
		if(_symbolValue != null) {
			return ""+_symbolValue;
		}
		if(!finite()) {
			return "Infinity";
		}
		if(nan()) {
			return "NaN";
		}
		return null
	}
	
	function toString():String {
		var s:String;
		if((s=specialString())!=null) {
			return s;
		}

		if(Math.abs(y)< 1e-15)
			return ""+x;
		var im:String;
		if(y<0) {
			im = (y==-1) ? "-i" : ""+y+"*i";
			return (Math.abs(x)<1e-15) ? im : "" + x +im;
		}
		else {
			im = (Math.abs(y-1)< 1e-15) ? "i" : ""+y+"*i";
			return (Math.abs(x)<1e-15) ? im : "" + x + "+" +im;
		}
	}
	
	function displayString():String {
		//
		// Display 3 sig fig only
		//
		var s:String;
		if((s=specialString())!=null) {
			return s;
		}

		var z:Complex = new Complex(toSigFig(x,3),toSigFig(y,3));
		return z.toString();
	}
	
	function toSigFig(t:Number, sigfigs:Number):Number {
		if(t==0) return t;
		var sign:Number=t/Math.abs(t);
		t = Math.abs(t);
		for(var k=0; t >= 10; k++) {
			t /= 10;
		}
		if(k==0) for(k=0; t < 1; k--) {
			t *= 10;
		}
		var n = (Math.pow(10,sigfigs-1));
		t = Math.round(t*n)/n;
		t *= (sign*Math.pow(10,k));
		return t;
	}
	
	// ------------------------------------------------------------------------
	
	//
	// Set evaluation functions (delegated)
	//
	function union(z:Complex):Complex {
		var sv:SetValue = setValue.union(z.setValue);
		//trace("creating new set: " + sv);
		return new Complex(sv);
	}
	
	// ------------------------------------------------------------------------
	static function test():Void {
		var a = new Complex(1,1);
		var b = new Complex(Math.sqrt(2), Math.PI/4, true);
		trace("sin(1,1) = "+ a.sin() + " sin(root2, pi/4) = " + b.sin());
		trace("|<3,4>|"+(new Complex(3,4)).length);
		trace("<1,1>*<1,1>="+a.times(b));
	}
}