﻿import Cons;
import DVector;
import DMatrix;
import Segment;

//  NOTE: The Group definitions in this file, and many of the comments,
//  came directly from the Mac version of Kali by Jeff Weeks.  Thanks
//  to Jeff for his lucid programming style and copius comments!
//    mbp Mon Sep 16 17:06:26 1996
/**
 * The SymmetryGroup class holds the data for a particular symmetry
 * group.  A symmetry group may be generated by
 *
 * <ul>
 *	<li>a reflection
 *	<li>a glide reflection
 *	<li>a rotation
 *	<li>up to two translations
 * </ul>
 * <p>
 * Kali always performs the generators in that order.  Given a line
 * segment upon which the group is to act, first the line segment is
 * reflected (if a reflection is present), then the line and its
 * reflection are glide reflected (if a glide reflection is present),
 * then the images are rotated (if a rotation is present) and finally
 * all images are translated according to whatever translations are
 * present.
 *
 * <p>
 * Note: all the math happens in the
 * <a href="Panorama.html#_top_">Panorama</a> object; the SymmetryGroup
 * object just holds the group data.
 *
 * @see SymmetryGroups
 * @see Panorama
 */

class SymmetryGroup {

	
/**
 * The index of the group (value is one of the constants defined
 * Constants.java)
 */
  public var index:Number;

/**
 * The name of the group (for debug printing, etc)
 */
  public var name:String;

/**
 * reflectionType has one of the values Cons.AXIS_NONE, Cons.AXIS_X0,
 * Cons.AXIS_X4, Cons.AXIS_Y0, or Cons.AXIS_Y4.  If reflectionType != Cons.AXIS_NONE,
 * do a reflection about the specified axis.
 */
  public var reflectionType:Number;

/**
 * If glideReflectionType != axis_none, do a glide reflection
 * a distance 1/2 along the specified axis.
 */
  public var glideReflectionType:Number;

/**
 * rotationOrder gives the order of the rotation about
 * the origin if a rotation is present.  Otherwise it's zero.
 */
  public var rotationOrder:Number;

/**
 * numTranslations tells how many independent translations are
 * present.  It will be 2 for a wallpaper group, 1 for a frieze
 * group, and 0 for a rosette group.  The array "translations"
 * gives the translation vectors.
 */
  public var numTranslations:Number;
  public var translations:Array;          // [2] of DVector



  /**
   * The number of mirror segments (same as mirrorSegments.length).
   */
  public var numMirrorSegments : Number;        // maximum is four
  /**
   * The locations of the mirror segments.  NOTE: currently
   * Java Kali does not display these; they're included here
   * because they were present in the Mac version, and a future
   * version of Java Kali might display them.
   */
  public var mirrorSegments:Array;     // [4] of Segments

  /**
   * XYZZYNOTE: currently
   * Java Kali does not display these; they're included here
   * because they were present in the Mac version, and a future
   * version of Java Kali might display them.
   */

  /**
   * The number of glide segments (same as mirrorSegments.length).
   */
  public var numGlideSegments:Number;         // maximum is two
  /**
   * The locations of the glide segments.
   */
  public var glideSegments:Array;      // [2] of Segments

  /**
   * The number of gyration points (same as mirrorSegments.length).
   */
 	public var numGyrationPoints:Number;        // maximum is four
  /**
   * The locations of the gyration points.
   */
  	public var gyrationPoints:Array;     // [4] of DVector

  /**
   * Create a new SymmetryGroup object with the given data 
   */
  	function SymmetryGroup(_index:Number,
                       		_name:String,
                      		_reflectionType:Number,
                       		_glideReflectionType:Number,
                       		_rotationOrder:Number,
                       		_translations : Array,
                       		_mirrorSegments : Array,
                       		_glideSegments : Array,
                       		_gyrationPoints : Array) {
      	index = _index;
     	name = _name;
    	reflectionType = _reflectionType;
    	glideReflectionType = _glideReflectionType;
    	rotationOrder = _rotationOrder;
    	numTranslations        = 0;
    	numMirrorSegments      = 0;
    	numGlideSegments       = 0;
    	numGyrationPoints      = 0;
		translations = _translations;
		if (translations != null) {
      		numTranslations      = translations.length;
		}
    	mirrorSegments = _mirrorSegments;
    	if (mirrorSegments != null) {
      		numMirrorSegments    = mirrorSegments.length;
    	}
    	glideSegments = _glideSegments;
    	if (glideSegments != null) {
      		numGlideSegments = glideSegments.length;
		}
    	gyrationPoints = gyrationPoints;
    	if (gyrationPoints != null) {
      		numGyrationPoints = gyrationPoints.length;
    	}
  	}

  /**
   * Convert an axis integer to a string for printing
   */
  private function axisString(axis:Number):String {
    switch (axis) {
      case Cons.AXIS_NONE: return ("AXIS_NONE");
      case Cons.AXIS_X0: return ("AXIS_X0");
      case Cons.AXIS_X4: return ("AXIS_X4");
      case Cons.AXIS_Y0: return ("AXIS_Y0");
      case Cons.AXIS_Y4: return ("AXIS_Y4");
      default: return ("???");
    }
  }
}