﻿//  NOTE: much of the code in this file, and many of the comments,
//  came directly from the Mac version of Kali by Jeff Weeks.  Thanks
//  to Jeff for his lucid programming style and copius comments!
//    mbp Mon Sep 16 17:06:26 1996
import Cons;
import SymmetryGroups;
import SymmetryGroup;
import DVector;
import KaliCanvas;
/**
 * The Panorama object is where the math happens in Kali.  It keeps
 * track of the current group, and applies that group's action to the
 * line segments as they are drawn.  (The DrawPanel object, which
 * stores the list of line segments, called the Panorama's
 * drawSegment() method once for each line segment.)
 *
 * @see DrawPanel
 * @see KaliCanvas
 */
class Panorama {

	/**
 	* The current group.
 	*/
	var symmetryGroups:SymmetryGroups;
	var symmetryGroup:SymmetryGroup;

	/**
 	* prepareToDraw() computes preparedTranslations, referenceVector,
 	* and maxReference, based on the current group and current
 	* coordinate system.
 	*/
  	var preparedTranslations:Array // = new DVector[2];
	var tOffsets:Array;
  	var referenceVector:DVector;
  	var maxReference:Number;

/**
 * The KaliCanvas object that we draw on.
 */
  	var kaliCanvas:KaliCanvas;

/**
 * Handy rotation matrices.
 */
 	var rotationMatrix:Array;

/**
 * Create a new Panorama object
 * @param kaliCanvas	The KaliCanvas object that this Panorama
 *			should use for drawing.
 */
	function Panorama(_kaliCanvas:KaliCanvas) {
    	kaliCanvas = _kaliCanvas;
		symmetryGroups = new SymmetryGroups();
		preparedTranslations = [];
		tOffsets = [0,0];
		
		// rotation Matrix is indexed by the group order of rotation
		rotationMatrix = [
    		new DMatrix(0.0, 0.0, 0.0, 0.0), // 0: unused
    		new DMatrix(1.0, 0.0, 0.0, 1.0), // 1: identity
    		new DMatrix(-1.0, 0.0, 0.0, -1.0), // 2: R180
    		new DMatrix(-0.5, -0.5*Cons.ROOT3, 0.5*Cons.ROOT3, -0.5), //3: R:-120
    		new DMatrix(0.0, -1.0, 1.0,  0.0), // 4: R:-90
    		new DMatrix(Cons.COS2PIOVER5, -Cons.SIN2PIOVER5, Cons.SIN2PIOVER5,  Cons.COS2PIOVER5), //5: R+72
    		new DMatrix(0.5, -0.5*Cons.ROOT3, 0.5*Cons.ROOT3,  0.5) //6 R:+60
    	];
  	}

/**
 * Set the current group.
 * @param index	The group index; should be one of the constants
 *		defined in Constants.java, e.g. Cons.GROUP_w2222 or
 *		Cons.GROUP_w3x3, etc.
 */
  public function setGroup(index:Number):Void {
	  trace("index = " + index);
    symmetryGroup = SymmetryGroup(symmetryGroups.group[index]);
  }

/**
 * Return the current group
 */
  public function getGroup():SymmetryGroup {
    return symmetryGroup;
  }


/**
 * Do the internal calculations necessary to prepare for drawing with
 * the current KaliCanvas coordinate systems.  This should be called
 * immediately before a sequence of drawSegment() calls to draw the
 * screen, but after the KaliCanvas object's coordinate systems have
 * been set.
 */
   public function prepareToDraw():Boolean {
    var i:Number;
    var area:Number;
    var width:Number;
	var altitude:Number;
	var det:Number;
	var dotProduct:Number;

    //	Transfer symmetryGroup's translations to the screen
    //	coordinate system.
    for (i = 0; i < symmetryGroup.numTranslations; i++) {
      preparedTranslations[i] = kaliCanvas.internalToScreen(symmetryGroup.translations[i]);
    }
    var pT0:DVector = DVector(preparedTranslations[0]);
    var pT1:DVector = DVector(preparedTranslations[1]);
	
    //	Make sure they're nondegenerate, and do further preparation as necessary.
    switch (symmetryGroup.numTranslations) {
    case 0:
      // No preparation is needed for a rosette group.
      return true;
      
    case 1:
      // Insist that the translation length be at least Cons.MIN_TRANSLATION pixels.
	  if (pT0.length() < Cons.MIN_TRANSLATION) {
		return false;
      }
      
      // If the translation vector is "mainly horizontal", make it point to the right.
      // If the translation vector is "mainly vertical",   make it point upwards.
      // Vectors on the 45 degree lines are considered horizontal.
      if (Math.abs(pT0.c[0]) >= Math.abs(pT0.c[1])) {
		if (pT0.c[0] < 0.0) {
		  pT0.negate();
		}
      } else {
		if (pT0.c[1] < 0.0) {
		  pT0.negate();
		}
      }
      return true;
      
    case 2:
      // Insist that the area of a fundamental parallelogram
      // be at least Cons.MIN_AREA, so that we don't end up drawing
      // 100,000 copies of a one-pixel image.
      area = Math.abs(pT0.c[0] * pT1.c[1] - pT0.c[1] * pT1.c[0]);
      if (area < Cons.MIN_AREA) {
		return false;
      }
      
      // In addition, insist that the parallelogram have altitude
      // at least Cons.MIN_ALTITUDE pixels in each direction, to avoid long
      // skinny parallelograms.
      for (i = 0; i < 2; i++) {
		width = (DVector(preparedTranslations[i])).length();
		altitude = area / width;
		if (altitude < Cons.MIN_ALTITUDE) {
		  return false;
		}
	  }
      
      // Let pT0 be the more horizontal pointing Vector.
      if (Math.abs(pT0.c[0]) < Math.abs(pT1.c[0])) {
			var temp:DVector = pT0.copy();
			pT0 = preparedTranslations[0] = pT1;
			pT1 = preparedTranslations[1] = temp;
      }
      
      // Make sure pT0 points to the right.
      if (pT0.c[0] < 0.0) {
		pT0.negate();
      }
      
      // Make sure preparedTranslations form a right-handed basis.
      det = pT0.c[0] * pT1.c[1] - pT0.c[1] * pT1.c[0];
      if (det < 0.0) {
		pT1.negate();
      }
      
	  // Add in users length adjustment
	  pT0.c[0] += tOffsets[0];
	  
      // Imagine ruling the window with lines parallel
      // to pT0.  To help us decide which
      // of these lines a given point lies on, we define a
      // reference vector such that
      // (1)	referenceVector is parallel to pT0, and
      // (2)	<referenceVector, pT1>.
      
      referenceVector = new DVector(-pT0.c[1], pT0.c[0]);
      
      dotProduct = referenceVector.dot(pT1);
      
      referenceVector.scale( 1.0 / dotProduct );
      
      // At this point we should have referenceVector[1] > 0.0
      
      // Find the maximum absolute value of <referenceVector, pt>
      // as pt ranges over the window frame.  Use the fact that the frame
      // is symmetrical about the origin.  The minimum will be the
      // negative of the maximum.
      maxReference = Math.abs(referenceVector.c[0]) * kaliCanvas.screenRight + Math.abs(referenceVector.c[1]) * kaliCanvas.screenTop;
      
      return true;
      
    default:
      return false;	//	should never occur
    }
  }

  function bases():Array {
	  return preparedTranslations;
  }
  
/**
 * Draw a single segment in the given colour under the
 * action of the current group.
 */
  public function drawSegment(s:Segment, c:Number):Void {
    drawSegmentRfGlRtTr(s, c);
  }

/** 
 * Draw the Segment under the action of symmetryGroup.
 * We take care of the possible reflection ("Rf") here,
 * and let other methods deal with the possible glide
 * reflection ("Gl"), rotation ("Rt") and translations ("Tr").
 */
  private function drawSegmentRfGlRtTr(s:Segment, c:Number):Void {

    //	Draw the Segment under the action of the possible
    //	glide reflection, rotation and translations.
    drawSegmentGlRtTr(s, c);

    //	If a reflection is called for, do it here, and then
    //	pass it to DrawSegmentGlRtTr() for the remaining operations.
    if (symmetryGroup.reflectionType != Cons.AXIS_NONE) {
      var reflectedSegment:Segment = s.copy();
	  var rsp0:DVector = reflectedSegment.p[0];
	  var rsp1:DVector = reflectedSegment.p[1];
      var i:Number;
      switch (symmetryGroup.reflectionType) {
        case Cons.AXIS_X0:
			rsp0.c[0] = - rsp0.c[0];
			rsp1.c[0] = - rsp1.c[0];
		  	break;
		case Cons.AXIS_X4:
			rsp0.c[0] = 0.5 - rsp0.c[0];
			rsp1.c[0] = 0.5 - rsp1.c[0];
		  /*
		  for (i = 0; i < 2; i++) {
			reflectedSegment.p[i].c[0] = 0.5 - reflectedSegment.p[i].c[0];
		  }
		  */
		  	break;
		case Cons.AXIS_Y0:
			rsp0.c[1] = - rsp0.c[1];
			rsp1.c[1] = - rsp1.c[1];
			/*
			  for (i = 0; i < 2; i++) {
				reflectedSegment.p[i].c[1] = - reflectedSegment.p[i].c[1];
			  }
			*/
			break;
		case Cons.AXIS_Y4:
			rsp0.c[1] = 0.5 - rsp0.c[1];
			rsp1.c[1] = 0.5 - rsp1.c[1];
			/*
		  	for (i = 0; i < 2; i++) {
				reflectedSegment.p[i].c[1] = 0.5 - reflectedSegment.p[i].c[1];
		  	}
			*/
		  	break;
		}
      	drawSegmentGlRtTr(reflectedSegment, c);
    }
  }

/**
 * Take care of the possible glide reflection ("Gl") here, and let
 * other methods deal with the possible rotation ("Rt") and
 * translations ("Tr").
 */
  private function drawSegmentGlRtTr(s:Segment, c:Number):Void {

    // Draw the Segment under the action of the possible
    // rotation and translations.
    drawSegmentRtTr(s, c);

    // If a glide reflection is called for, do it here, and then
    // pass it to DrawSegmentRtTr() for the remaining operations.
    if (symmetryGroup.glideReflectionType != Cons.AXIS_NONE) {
      var glideSegment:Segment = s.copy();
	  var gsp0:DVector = glideSegment.p[0];
	  var gsp1:DVector = glideSegment.p[1];
      var i:Number;
      switch (symmetryGroup.glideReflectionType) {
      	case Cons.AXIS_X0:
		  gsp0.c[0] = -gsp0.c[0];
		  gsp0.c[1] += 0.5;
		  gsp1.c[0] = -gsp1.c[0];
		  gsp1.c[1] += 0.5;
		  /*
		  for (i = 0; i < 2; i++) {
			glideSegment.p[i].c[0] = - glideSegment.p[i].c[0];
			glideSegment.p[i].c[1] += 0.5;
		  }
		  */
		  break;
		case Cons.AXIS_X4:
		  gsp0.c[0] = 0.5 - gsp0.c[0];
		  gsp0.c[1] += 0.5;
		  gsp1.c[0] = 0.5 - gsp1.c[0];
		  gsp1.c[1] += 0.5;
		  /*
		  for (i = 0; i < 2; i++) {
			glideSegment.p[i].c[0] = 0.5 - glideSegment.p[i].c[0];
			glideSegment.p[i].c[1] += 0.5;
		  }
		  */
		  break;
		case Cons.AXIS_Y0:
		  gsp0.c[1] = -gsp0.c[1];
		  gsp0.c[0] += 0.5;
		  gsp1.c[1] = -gsp1.c[1];
		  gsp1.c[0] += 0.5;
		  /*
		  for (i = 0; i < 2; i++) {
			glideSegment.p[i].c[1] = - glideSegment.p[i].c[1];
			glideSegment.p[i].c[0] += 0.5;
		  }
		  */
		  break;
		case Cons.AXIS_Y4:
		  gsp0.c[1] = 0.5 - gsp0.c[1];
		  gsp0.c[0] += 0.5;
		  gsp1.c[1] = 0.5 - gsp1.c[1];
		  gsp1.c[0] += 0.5;
		  /*
		  for (i = 0; i < 2; i++) {
			glideSegment.p[i].c[1] = 0.5 - glideSegment.p[i].c[1];
			glideSegment.p[i].c[0] += 0.5;
		  }
		  */
		  break;
		}
      drawSegmentRtTr(glideSegment, c);
    }

  }

/**
 * Take care of the possible rotation ("Rt") here, and let
 * another method deal with the possible translations ("Tr").
 */
  private  function drawSegmentRtTr(s:Segment, c:Number):Void {

    var rotatedSegment:Segment = s.copy();
    var i:Number;
	var j:Number;
    for (i = 0; i < symmetryGroup.rotationOrder; i++) {
      drawSegmentTr(rotatedSegment, c);
      for (j = 0; j < 2; j++) {
		rotatedSegment.p[j] = DMatrix(rotationMatrix[symmetryGroup.rotationOrder]).timesV(DVector(rotatedSegment.p[j]));
	  }
    }
  }

/**  
 * Draw a Segment under the action of symmetryGroup's translations.
 */
  private function drawSegmentTr(s:Segment, c:Number):Void {
    
    // Here we must finally leave the pristine world of the internal
    // coordinate system and deal with the realities of the window
    // coordinate system.

    var screenSegment:Segment = new Segment(kaliCanvas.internalToScreen(s.p[0]),
					kaliCanvas.internalToScreen(s.p[1]));

    // Break into separate cases for rosette, frieze and wallpaper groups.
    switch (symmetryGroup.numTranslations) {
      case 0:
        drawSegmentRosette(screenSegment, c);
		break;
      case 1:
		drawSegmentFrieze(screenSegment, c);
		break;
      case 2:
		drawSegmentWallpaper(screenSegment, c);
		break;
      }
  }


  private function drawSegmentRosette(s:Segment, c:Number):Void {
    kaliCanvas.drawSegment(s, c);
  }

  private function drawSegmentFrieze(segment:Segment,  colour:Number):Void {
    var	i:Number;
    var j:Number;
    var multiple:Number;
    var	translatedSegment:Segment;

    //	At the moment I can't think of a more elegant way to do it,
    //	so let's break into cases according to whether the translation
    //	is mainly horizontal or mainly vertical.
    //	[prepareToDraw() has checked that it's not
    //	completely degenerate.]
	var pT0:DVector = preparedTranslations[0];
	var pT1:DVector = preparedTranslations[1];
	
    if (Math.abs(pT0.c[0]) >= Math.abs(pT0.c[1])) { // mainly horiz
      // mainly horizontal
      //
      // prepareToDraw() has already made sure that
      // pT0 points to the right.

      // Copy segment to translatedEndpoints in such a way that
      // translatedSegment.p[0] is not to the right of
      // translatedSegment.p[1].
      translatedSegment = segment.copy();
      if (DVector(segment.p[0]).c[0] > DVector(segment.p[1]).c[0]) {
		translatedSegment.reverse();
      }

      // Subtract off some multiple of pT0 to move
      // to the left most image which might be visible.
	  var tsp0:DVector = translatedSegment.p[0];
	  var tsp1:DVector = translatedSegment.p[1];
      multiple = Math.floor((tsp1.c[0] - kaliCanvas.screenLeft) / pT0.c[0]);
      //for (i = 0; i < 2; i++) {
		for (j = 0; j < 2; j++) {
			var m:Number = multiple * pT0.c[j];
	  		tsp0.c[j] -= m;
	  		tsp1.c[j] -= m;
	  		//translatedSegment.p[i].c[j] -= multiple * pT0.c[j];
		}
      //}

      // Draw images until we move off the right side of the frame.
      while (tsp0.c[0] < kaliCanvas.screenRight) {

		kaliCanvas.drawSegment(translatedSegment, colour);

		//for (i = 0; i < 2; i++) {
		  for (j = 0; j < 2; j++) {
			tsp0.c[j] += pT0.c[j];
			tsp1.c[j] += pT0.c[j];
			//translatedSegment.p[i].c[j] += pT0.c[j];
		  }
		//}
	  }
    } else {
      // mainly vertical
      //
      // prepareToDraw() has already made sure that
      // pT0 points upward.

      // Copy endpoints to translatedSegment.p in such a way that
      // translatedSegment.p[0] is not above translatedSegment.p[1].
      translatedSegment = segment.copy();
      if (DVector(segment.p[0]).c[1] > DVector(segment.p[1]).c[1]) {
		translatedSegment.reverse();
      }

	  var tsp0:DVector = translatedSegment.p[0];
	  var tsp1:DVector = translatedSegment.p[1];

	  // Subtract off some multiple of pT0 to move
      // to the lowest image which might be visible.
      multiple = Math.floor((tsp1.c[1] - kaliCanvas.screenBottom) / pT0.c[1]);
      //for (i = 0; i < 2; i++) {
		for (j = 0; j < 2; j++) {
			var m:Number = multiple * pT0.c[j];
	  		tsp0.c[j] -= m;
	  		tsp1.c[j] -= m;
	  		//translatedSegment.p[i].c[j] -= multiple * pT0.c[j];
		}
      //}

      // Draw images until we move off the top of the frame.
      while (tsp0.c[1] < kaliCanvas.screenTop) {
		kaliCanvas.drawSegment(translatedSegment, colour);
		//for (i = 0; i < 2; i++) {
	  		for (j = 0; j < 2; j++) {
				tsp0.c[j] += pT0.c[j];
				tsp1.c[j] += pT0.c[j];
	    		//translatedSegment.p[i].c[j] += pT0.c[j];
	  		}
		//}
      }
    }

  }

  private function drawSegmentWallpaper(segment:Segment,  colour:Number):Void {
    
    var	i:Number;
    var j:Number;
    var numRows:Number;
    var row:Number;
    var dotProduct:Array = [] // new double[2];
    var higherDotProduct:Number;
    var dotProductDifference:Number;
    var multiple:Number;
    var translatedSegment:Segment;
	var pT0:DVector = DVector(preparedTranslations[0]);
	var pT1:DVector = DVector(preparedTranslations[1]);
    
    //	prepareToDraw() has already set up preparedTranslations so that
    //	(1)	pT0 points to the right,
    //	(2)	preparedTranslations[] for a right-handed basis, and
    //	(3)	the corresponding parallelogram has Cons.MIN_AREA and Cons.MIN_ALTITUDE.
    
    //	We want to draw all translates whose initial endpoint lies within the frame.
    
    //	Copy segment to translatedSegment in such a way that
    //	translatedSegment.p[0] is not to the right of translatedSegment.p[1].
    translatedSegment = segment.copy();
    if (DVector(segment.p[0]).c[0] > DVector(segment.p[1]).c[0]) {
      translatedSegment.reverse();
    }
    
	var tsp0:DVector = translatedSegment.p[0];
	var tsp1:DVector = translatedSegment.p[1];

	//	Imagine the coordinate grid defined by preparedTranslations[].
    //	Compute <translatedSegment.p[i], referenceVector> to see which horizontal
    //	grid line we're near (cf. the definition of referenceVector in
    //	prepareToDraw()).  E.g. if dotProduct is 2.25, we're
    //	a quarter of the way from grid line 2 to grid line 3.
    for (i = 0; i < 2; i++) {
      dotProduct[i] = referenceVector.dot(DVector(translatedSegment.p[i]));
    }
    
    higherDotProduct = Math.max(dotProduct[0], dotProduct[1]);
    dotProductDifference = Math.abs(dotProduct[0] - dotProduct[1]);
    
    //	Subtract off some multiple of pT1 so that
    //	we move to the lowest parallel line which might still intersect the window.
    multiple = Math.floor(higherDotProduct - (-maxReference));
    //for (i = 0; i < 2; i++) {
      for (j = 0; j < 2; j++) {
		var m:Number = multiple * pT1.c[j];
	  	tsp0.c[j] -= m;
	  	tsp1.c[j] -= m;
		//translatedSegment.p[i].c[j] -= multiple * pT1.c[j];
      }
    //}
    
    //	The number of rows we need to consider is given by
    //	[maxReference - (-maxReference)] + dotProductDifference
    //	= 2*maxReference + dotProductDifference.
    numRows = Math.round(Math.ceil(2.0*maxReference + dotProductDifference));
    
    for (row = 0; row < numRows; row++) {
      //	Draw one row.
      //	Subtract some multiple of pT0 to move
      //	to the left most position within the row.
      multiple = Math.floor((tsp1.c[0] - kaliCanvas.screenLeft) / pT0.c[0]);
      //for (i = 0; i < 2; i++) {
		for (j = 0; j < 2; j++) {
			var m:Number = multiple * pT0.c[j];
	  		tsp0.c[j] -= m;
	  		tsp1.c[j] -= m;
		  //translatedSegment.p[i].c[j] -= multiple * pT0.c[j];
		}
      //}
      
      //	Draw images until we move off the right side of the frame.
      while (tsp0.c[0] < kaliCanvas.screenRight) {

		kaliCanvas.drawSegment(translatedSegment, colour);
	
		//for (i = 0; i < 2; i++) {
		  for (j = 0; j < 2; j++) {
			tsp0.c[j] += pT0.c[j];
			tsp1.c[j] += pT0.c[j];
			//translatedSegment.p[i].c[j] += pT0.c[j];
		  }
		//}
      }
      
      //	Add pT1 to each of translatedSegment
      //	to move up to the next row.
      //for (i = 0; i < 2; i++) {
		for (j = 0; j < 2; j++) {
			tsp0.c[j] += pT1.c[j];
			tsp1.c[j] += pT1.c[j];
		  //translatedSegment.p[i].c[j] += pT1.c[j];
		}
      //}
    }
  }
  
}
