﻿/*
Subject: 22) How do I split a bezier at a specific value for t?

    A Bezier curve is a parametric polynomial function from the
    interval [0..1] to a space, usually 2-D or 3-D.  Common Bezier
    curves use cubic polynomials, so have the form

        f(t) = a3 t^3 + a2 t^2 + a1 t + a0,

    where the coefficients are points in 3-D. Blossoming converts this
    polynomial to a more helpful form.  Let s = 1-t, and think of
    tri-linear interpolation:

        F([s0,t0],[s1,t1],[s2,t2]) =
            s0(s1(s2 c30 + t2 c21) + t1(s2 c21 + t2 c12)) +
            t0(s1(s2 c21 + t2 c12) + t1(s2 c12 + t2 c03))
            =
            c30(s0 s1 s2) +
            c21(s0 s1 t2 + s0 t1 s2 + t0 s1 s2) +
            c12(s0 t1 t2 + t0 s1 t2 + t0 t1 s2) +
            c03(t0 t1 t2).

    The data points c30, c21, c12, and c03 have been used in such a
    way as to make the resulting function give the same value if any
    two arguments, say [s0,t0] and [s2,t2], are swapped. When [1-t,t]
    is used for all three arguments, the result is the cubic Bezier
    curve with those data points controlling it:

          f(t) = F([1-t,t],[1-t,t],[1-t,t])
               =  (1-t)^3 c30 +
                 3(1-t)^2 t c21 +
                 3(1-t) t^2 c12 +
                 t^3 c03.
*/
function bezier3(t:Number, a:Object, b:Object, c:Object, d:Object):Object {
	//trace("bezier3("+w+")");
	var t2 = t*t;
	var t3 = t*t2;
	var u = 1-t;
	var u2 = u*u;
	var u3 = u*u2;
	var x = t3*b._x+3*t2*u*d._x+3*t*u2*c._x+u3*a._x;
	var y = t3*b._y+3*t2*u*d._y+3*t*u2*c._y+u3*a._y;
	return {_x:x, _y:y};
}
/*
    Notice that
           F([1,0],[1,0],[1,0]) = c30,
           F([1,0],[1,0],[0,1]) = c21,
           F([1,0],[0,1],[0,1]) = c12, _
           F([0,1],[0,1],[0,1]) = c03.

    In other words, cij is obtained by giving F argument t's i of
    which are 0 and j of which are 1. Since F is a linear polynomial
    in each argument, we can find f(t) using a series of simple steps.
    Begin with

        f000 = c30, f001 = c21, f011 = c12, f111 = c03.

    Then compute in succession:

        f00t = s f000 + t f001, f01t = s f001 + t f011,
        f11t = s f011 + t f111,
        f0tt = s f00t + t f01t, f1tt = s f01t + t f11t,
        fttt = s f0tt + t f1tt.
*/
function bezierSplit(t:Number, a:Number, b:Number, c:Number, d:Number):Array {
	var f000 = a;
	var f001 = c;
	var f011 = d;
	var f111 = b;
	var s = 1-t;
	var f00t = s*f000+t*f001;
	var f01t = s*f001+t*f011;
	var f11t = s*f011+t*f111;
	var f0tt = s*f00t+t*f01t;
	var f1tt = s*f01t+t*f11t;
	var fttt = s*f0tt+t*f1tt;
	return [{a:f000, c:f00t, d:f0tt, b:fttt}, {a:fttt, c:f1tt, d:f11t, b:f111}];
}
/*
    This is the de Casteljau algorithm for computing f(t) = fttt.

    It also has split the curve for the intervals [0..t] and [t..1].
    The control points for the first interval are f000, f00t, f0tt,
    fttt, while those for the second interval are fttt, f1tt, f11t,
    f111.

    If you evaluate 3 F([1-t,t],[1-t,t],[-1,1]) you will get the
    derivate of f at t. Similarly, 2*3 F([1-t,t],[-1,1],[-1,1]) gives
    the second derivative of f at t, and finally using 1*2*3
    F([-1,1],[-1,1],[-1,1]) gives the third derivative.

    This procedure is easily generalized to different degrees,
    triangular patches, and B-spline curves.
*/
